<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PageResource\Pages;
use App\Filament\Resources\PageResource\RelationManagers;
use App\Models\Page;
use Filament\Forms;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Mail\Markdown;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;

class PageResource extends Resource
{
    protected static ?string $model = Page::class;
    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = 20;
    protected static ?string $navigationGroup = 'Tartalom';

    public static function getModelLabel(): string {
        return __('admin.page');
    }

    public static function getPluralModelLabel(): string {
        return __('admin.pages');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Group::make()
                    ->schema([
                        Section::make()
                            ->columns(2)
                            ->schema([
                                TextInput::make('title')
                                    ->label(Lang::get('messages.title'))
                                    ->required()
                                    ->live(onBlur:true)
                                    ->afterStateUpdated(function(string $operation, $state, Forms\Set $set) {
                                        if($operation !== 'create') {
                                            return;
                                        }
                                        $set('slug', Str::slug($state));
                                    }),
                                TextInput::make('slug')
                                    ->label(Lang::get('messages.slug'))
                                    ->prefix('page/')
                                    ->disabled()
                                    ->required()
                                    ->dehydrated()
                                    ->unique(Page::class, 'slug', ignoreRecord: true),
                                Textarea::make('description')
                                    ->label(Lang::get('messages.description'))
                                    ->columnSpanFull(),
                                MarkdownEditor::make('content')
                                    ->label(Lang::get('messages.content'))
                                    ->columnSpanFull(),
                            ]),
                    ]),
                Group::make()
                    ->schema([
                        Section::make()
                            ->columns(2)
                            ->schema([
                                DateTimePicker::make('created_at')
                                    ->label(Lang::get('messages.created_at'))
                                    ->disabled(),
                                DateTimePicker::make('updated_at')
                                    ->label(Lang::get('messages.updated_at'))
                                    ->disabled(),
                                Toggle::make('is_active')
                                    ->label(Lang::get('messages.is_active')),
                            ]),
                        Section::make()
                            ->columns(2)
                            ->schema([
                                TextInput::make('meta_title')
                                    ->label(Lang::get('messages.meta_title')),
                                TextInput::make('meta_author')
                                    ->label(Lang::get('messages.meta_author')),
                                Textarea::make('meta_description')
                                    ->label(Lang::get('messages.meta_description'))
                                    ->columnSpanFull(),
                                TextInput::make('meta_keywords')
                                    ->label(Lang::get('messages.meta_keywords'))
                                    ->columnSpanFull(),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title')
                    ->label(Lang::get('messages.title'))
                    ->searchable()
                    ->sortable(),
                TextColumn::make('slug')
                    ->label(Lang::get('messages.slug'))
                    ->prefix('page/')
                    ->sortable()
                    ->toggleable(),
                IconColumn::make('is_active')
                    ->label(Lang::get('messages.is_active'))
                    ->boolean(),
                TextColumn::make('created_at')
                    ->label(Lang::get('messages.created_at'))
                    ->toggleable()
                    ->toggledHiddenByDefault(),
                TextColumn::make('updated_at')
                    ->label(Lang::get('messages.updated_at'))
                    ->toggleable()
                    ->toggledHiddenByDefault(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPages::route('/'),
            'create' => Pages\CreatePage::route('/create'),
            'edit' => Pages\EditPage::route('/{record}/edit'),
        ];
    }
}
