<?php

namespace App\Filament\Resources;

use App\Enums\RoleEnum;
use App\Filament\Resources\UserResource\Pages;
use App\Filament\Resources\UserResource\Pages\CreateUser;
use App\Models\User;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Pages\Page;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Lang;

class UserResource extends Resource
{
    protected static ?string $model = User::class;
    protected static ?string $navigationIcon = 'heroicon-o-users';
    protected static ?string $recordTitleAttribute = 'email';
    protected static ?int $navigationSort = 52;
    protected static ?string $navigationGroup = 'Beállítások';

    public static function getModelLabel(): string {
        return __('messages.user');
    }

    public static function getPluralModelLabel(): string {
        return __('messages.users');
    }


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Group::make()
                    ->schema([
                        Section::make()
                            ->schema([
                                TextInput::make('name')
                                    ->label(Lang::get('messages.name'))
                                    ->required(),
                                TextInput::make('email')
                                    ->label(Lang::get('messages.email'))
                                    ->required(),
                                TextInput::make('password')
                                    ->label(Lang::get('messages.password'))
                                    ->password()
                                    ->required(fn(Page $livewire): bool => $livewire instanceof CreateUser)
                                    ->minLength(6)
                                    ->same('passwordComfirmation')
                                    ->dehydrated(fn($state) => filled($state))
                                    ->dehydrateStateUsing(fn($state) => Hash::make($state)),
                                TextInput::make('passwordComfirmation')
                                    ->label(Lang::get('messages.password_comfirmation'))
                                    ->password()
                                    ->required(fn(Page $livewire): bool => $livewire instanceof CreateUser)
                                    ->minLength(6)
                                    ->dehydrated(false)

                            ])
                    ]),
                Group::make()
                    ->schema([
                        Section::make()
                            ->schema([
                                Select::make('role')
                                    ->label(Lang::get('messages.role'))
                                    ->options([
                                        RoleEnum::ADMIN->value => 'Admin',
                                        RoleEnum::USER->value => 'User'
                                    ])
                            ])
                    ])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label(Lang::get('messages.name'))
                    ->searchable()
                    ->sortable(),
                TextColumn::make('email')
                    ->label(Lang::get('messages.email'))
                    ->searchable()
                    ->sortable(),
                TextColumn::make('role')
                    ->label(Lang::get('messages.role'))
                    ->badge()
                    ->sortable()
                    ->color(fn (string $state) => match($state) {
                        'root' => Color::Amber,
                        'admin' => Color::Red,
                        'user' => Color::Blue
                    }),
                TextColumn::make('created_at')
                    ->label(Lang::get('messages.created_at'))
                    ->since()
                    ->sortable(),
                TextColumn::make('updated_at')
                    ->label(Lang::get('messages.updated_at'))
                    ->since()
                    ->sortable(),
            ])
            ->filters([
                SelectFilter::make('role')
                    ->label(Lang::get('messages.role'))
                    ->multiple()
                    ->options([
                        RoleEnum::ROOT->value => 'Root',
                        RoleEnum::ADMIN->value => 'Admin',
                        RoleEnum::USER->value => 'User'
                    ])
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('id', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }
}
