<?php

namespace App\Services;

use App\Models\Cart;
use App\Models\CartFood;
use App\Models\Config;
use App\Models\Coupon;
use App\Models\Food;
use App\Models\Payment;
use App\Models\User;
use App\Models\Zip;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;

class CartService {
    private Cart $cart;

    public function __construct(
        private Request $request,
        private ?User $user) {
            $this->cart = $this->initCart();
    }

    public function add(Food $food): void {
        $cartFood = CartFood::where('cart_id', $this->cart->id)
                            ->where('food_id', $food->id)->first();

        if($cartFood == null) {
            $cartFood = new CartFood();
            $cartFood->cart_id = $this->cart->id;
            $cartFood->food_id = $food->id;
            $cartFood->quantity = 1;
            $cartFood->price = $food->price;
            $cartFood->save();
        } else {
            $this->increase($food);
        }
    }

    public function remove(Food $food): void {
        $cartFood = CartFood::where('cart_id', $this->cart->id)
                            ->where('food_id', $food->id)->first();
        $cartFood->delete();
    }

    public function increase(Food $food) {
        $cartFood = CartFood::where('cart_id', $this->cart->id)
                            ->where('food_id', $food->id)->first();
        $cartFood->quantity++;
        $cartFood->save();
    }

    public function decrease(Food $food) {
        $cartFood = CartFood::where('cart_id', $this->cart->id)
                            ->where('food_id', $food->id)->first();
        $cartFood->quantity--;
        $cartFood->save();

        if($cartFood->quantity <= 0) {
            $cartFood->delete();
        }
    }

    public function delete(Food $food) {
        $cartFood = CartFood::where('cart_id', $this->cart->id)
                            ->where('food_id', $food->id)->first();
        $cartFood->delete();
    }

    public function getQuantity(Food $food): int {
        $cartFood = CartFood::where('cart_id', $this->cart->id)
                            ->where('food_id', $food->id)->first();
        $quantity = 1;

        if($cartFood instanceof CartFood) {
            $quantity = $cartFood->quantity;
        }

        return $quantity;
    }

    public function getFoods(): Collection {
        $cartFoods = CartFood::where('cart_id', $this->cart->id)->get();

        foreach ($cartFoods as $value) {
            $this->checkClosedFood($value);
        }

        return $cartFoods;
    }

    public function getTotal(): array {
        $cartFoods = $this->getFoods();
        $total = [
            'total' => 0,
            'food' => 0,
            'quantity_discount' => 0,
        ];

        foreach ($cartFoods as $food) {
            $total['food'] += ($food->price * $food->quantity);
        }

        $total['quantity_discount'] = DiscountService::getCartQuantityDiscount($cartFoods);
        $total['total'] += $total['food'];
        $total['total'] -= $total['quantity_discount'];

        return $total;
    }

    public function getAllTotal(?Payment $payment, $delivery_zip, ?Coupon $coupon): Collection {
        $cartFoods = $this->getFoods();
        $packCount = 0;
        $total = [
            'items' => [],
            'food' => 0,
            'quantity_discount' => 0,
            'pack' => 0,
            'delivery' => 0,
            'payment' => 0,
            'coupon' => 0,
            'total' => 0
        ];

        foreach ($cartFoods as $item) {
            $packCount += $item->quantity;
            $total['items'][$item->id] = $item->quantity * $item->price;
            $total['food'] += $total['items'][$item->id];
        }

        $total['quantity_discount'] = DiscountService::getCartQuantityDiscount($cartFoods);

        $packUnitPrice = Config::where('name', 'pack_unit_price')->first();
        if($packUnitPrice != null) {
            if($packUnitPrice > 0) {
                $total['pack'] = $packCount * $packUnitPrice->value;
            }
        }

        if($delivery_zip != null) {
            $zip = Zip::where('zip', $delivery_zip)->first();
            if($zip != null) {
                if($zip->price > 0) {
                    $total['delivery'] = ShippingService::getShippingPrice($zip, $cartFoods);
                }
            }
        }

        if($payment != null) {
            if($payment->price > 0) {
                $total['payment'] = $payment->price;
            }
        }

        if($coupon instanceof Coupon){
            if(CouponService::validate($coupon->code)) {
                $total['coupon'] = $coupon->value;
            }
        }

        $total['total'] += $total['food'];
        $total['total'] += $total['pack'];
        $total['total'] += $total['delivery'];
        $total['total'] += $total['payment'];
        $total['total'] -= $total['coupon'];
        $total['total'] -= $total['quantity_discount'];

        return new Collection($total);
    }

    public function completed(): void {
        $this->cart->completed_at = new DateTime();
        $this->cart->save();
    }

    private function initCart(): Cart {
        $sessionToken = $this->request->session()->get('_token');

        $cart = Cart::where('token', $sessionToken)
                    ->where('deleted_at', null)
                    ->where('completed_at', null)
                    ->first();

        if( $cart != null && $this->user != null ) {
            if($cart->user == null) {
                $cart->user = $this->user->id;
                $cart->save();
            }
        }

        if( $cart == null && $this->user != null ) {
            $cart = Cart::where('user', $this->user->id)
                        ->where('completed_at', null)
                        ->where('deleted_at', null)
                        ->first();

            if($cart != null) {
                $cart->token = $sessionToken;
                $cart->save();
            }
        }

        if( $cart == null ) {
            $cart = new Cart();
            $cart->token = $sessionToken;

            if($this->user instanceof User) {
                $cart->user = $this->user->id;
            }
            $cart->save();
        }

        return $cart;
    }

    private function checkClosedFood(CartFood $cartFood): void {
        $food = Food::where('id', $cartFood->food_id)->first();

        if($food->closing_at < now()) {
            $cartFood->delete();
        }
    }
}