<?php

namespace App\Services;

use App\Enums\OrderPaymentStatusEnum;
use App\Models\Order;
use App\Services\OrderService;
use App\Services\SimplePay\ResultStatusType;
use App\Services\SimplePay\SimplePayBack;
use App\Services\SimplePay\SimplePayIpn;
use App\Services\SimplePay\SimplePayStart;
use Illuminate\Database\Eloquent\Casts\Json;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SimplePayService {

    public static function start(Order $order): string {
        $config = self::getConfig();
        $price = OrderService::getTotalPrice($order);

        $trx = new SimplePayStart;
        $trx->addData('currency', 'HUF');
        $trx->addConfig($config);
        $trx->addData('total', $price['total']);
        $trx->addData('orderRef', str_replace(array('.', ':', '/'), "", @$_SERVER['SERVER_ADDR']) . @date("U", time()) . rand(1000, 9999));
        $trx->addData('customer', $order->fullname);
        $trx->addData('customerEmail', $order->email);
        $trx->addData('language', 'HU');

        $timeoutInSec = 600;
        $timeout = @date("c", time() + $timeoutInSec);
        $trx->addData('timeout', $timeout);

        $trx->addData('methods', array('CARD'));
        $trx->addData('url', $config['URL']);

        $trx->runStart();

        $order->simplepay_order_ref = $trx->getReturnData()["orderRef"];
        $order->simplepay_transaction_id = $trx->getReturnData()["transactionId"];
        $order->save();

        return $trx->getReturnData()["paymentUrl"];
    }

    public static function back(Request $request): array {
        $config = self::getConfig();

        $trx = new SimplePayBack;
        $trx->addConfig($config);

        $result = array();
        if ($request->get('r') != null && $request->get('s') != null) {
            if ($trx->isBackSignatureCheck($request->get('r'), $request->get('s'))) {
                $result = $trx->getRawNotification();
            }
        }

        $status = $result["e"];
        $simplePayOrderRef = $result["o"];
        $simplePayTransactionId = $result["t"];

        $order = Order::where('simplepay_order_ref', $simplePayOrderRef)
                        ->where('simplepay_transaction_id', $simplePayTransactionId)
                        ->first();

        if($status == ResultStatusType::SUCCESS) {
            $order->payment_status = OrderPaymentStatusEnum::SUCCESS;
            $order->save();
            return [
                'status' => true,
                'order' => $order
            ];
        } elseif ($status == ResultStatusType::FAIL) {
            $order->payment_status = OrderPaymentStatusEnum::FAIL;
        } elseif ( $status == ResultStatusType::CANCEL ) {
            $order->payment_status = OrderPaymentStatusEnum::CANCEL;
        } elseif ( $status == ResultStatusType::TIMEOUT ) {
            $order->payment_status = OrderPaymentStatusEnum::TIMEOUT;
        } else {
            $order->payment_status = OrderPaymentStatusEnum::ERROR;
        }

        $order->save();
        return [
            'status' => false,
            'order' => $order
        ];
    }

    public static function ipn(Request $request) {
        $config = self::getConfig();
        $trx = new SimplePayIpn;
        $trx->addConfig($config);

        $parameters = $request->request->all();
        $json = json_encode($parameters);

        if (!$trx->isIpnSignatureCheck($json)) {
            $confirm = $trx->getIpnConfirmContent();

            $order = Order::where('simplepay_order_ref', $parameters['orderRef'])
                        ->where('simplepay_transaction_id', $parameters['transactionId'])
                        ->first();

            if($parameters['status'] === "FINISHED") {
                $order->payment_status = OrderPaymentStatusEnum::SUCCESS->value;
            } elseif ($parameters['status'] === "CANCELED") {
                $order->payment_status = OrderPaymentStatusEnum::CANCEL->value;
            } elseif ($parameters['status'] === "TIMEOUT") {
                $order->payment_status = OrderPaymentStatusEnum::TIMEOUT->value;
            } else {
                $order->payment_status = OrderPaymentStatusEnum::FAIL->value;
            }

            $order->save();
            $trx->runIpnConfirm();
        }
    }

    private static function getConfig(): array {
        return [
            'HUF_MERCHANT' => env('SIMPLEPAY_HUF_MERCHANT'),
            'HUF_SECRET_KEY' => env('SIMPLEPAY_HUF_SECRET_KEY'),
            'EUR_MERCHANT' => "",
            'EUR_SECRET_KEY' => "",
            'USD_MERCHANT' => "",
            'USD_SECRET_KEY' => "",
            'SANDBOX' => env('SIMPLEPAY_SANDBOX') == 'true'? true : false,
            'URL' => 'http://' . env('HTTP_HOST') . '/api/simplepay/back',
            'GET_DATA' => (isset($_GET['r']) && isset($_GET['s'])) ? ['r' => $_GET['r'], 's' => $_GET['s']] : [],
            'POST_DATA' => $_POST,
            'SERVER_DATA' => $_SERVER,
            'LOGGER' => true,
            'LOG_PATH' => 'log',
            'AUTOCHALLENGE' => true,
        ];
    }

}